/**************************************************************************************
  $Id: cifXTCPServerDlg.cpp 13831 2020-12-01 08:53:21Z RMayer $

  Description:
    Server handling

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2020-12-01  Make xChannelIOInfo() callable by adding it to the tDRVFunctions table
    2019-04-04  Add support for xSysdeviceResetEx()
    2015-09-22  Fix prototype of Marshaller Timer
    2013-03-14  Added support for xDriverRestartDevice()
    2010-12-13  Download hook added
    2010-01-29  - Add FindFirstFile/FindNextFile driver functions to marshaller configuration
                - Add doxygen comments
    2009-12-04  - Wrapper for xSysdeviceOpen(),xSysdeviceClose()
                  xChannelOpen(), xChannelClose() added to track channel access
                - Name of connected client added to list
                - GUI design changed
                - Use CTreeListCtrl to display clients
    ????-??-??  created (unknown source)

 **************************************************************************************/

#include "stdafx.h"
#include "cifXTCPServer.h"
#include "cifXTCPServerDlg.h"

#include "MarshallerConfig.h"
#include "MarshallerVersion.h"
#include "cifXTransport.h"
#include "TCPConnector.h"
#include "cifXErrors.h"
#include "verinfo.h"
#include "HostInformation.h"
#include "cifXDownloadHook.h"
#include "netXAPI.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#endif

/*****************************************************************************/
/*! Translate cifX device handle to NX-API handle                            */
/*****************************************************************************/
static CIFXHANDLE GetNxApiHandle(uint32_t ulSerialNumber, uint32_t ulDeviceNumber)
{
  NXDRV_DEVICE_INFORMATION         tDeviceInfo      = {0};
  uint32_t                         ulSearchIdx      = 0;
  CIFXHANDLE                       hRet             = NULL;

  if (NXAPI_NO_ERROR == nxDrvFindDevice( NXDRV_FIND_FIRST, sizeof(tDeviceInfo), &tDeviceInfo, &ulSearchIdx))
  {
    if ( (tDeviceInfo.tSystemInfoBlock.ulSerialNumber == ulSerialNumber) &&
         (tDeviceInfo.tSystemInfoBlock.ulDeviceNumber == ulDeviceNumber)   )
    {
      hRet = tDeviceInfo.hDevice;
    } else
    {
      while (NXAPI_NO_ERROR == nxDrvFindDevice( NXDRV_FIND_NEXT, sizeof(tDeviceInfo), &tDeviceInfo, &ulSearchIdx))
      {
        if ( (tDeviceInfo.tSystemInfoBlock.ulSerialNumber == ulSerialNumber) &&
             (tDeviceInfo.tSystemInfoBlock.ulDeviceNumber == ulDeviceNumber)   )
        {
          hRet = tDeviceInfo.hDevice;
          break;
        }
      }
    }
  }

  return hRet;
}

/*****************************************************************************/
/*! File storage callback (uses NX-API)                                      */
/*****************************************************************************/
static int32_t HandleFileStorage(BOARD_INFORMATION* ptBoardInfo,
                                 char* pszFileName, uint32_t ulFileSize,
                                 uint8_t* pabFileData, uint32_t ulChannel,
                                 uint32_t ulDownloadMode, void* pvUser)
{
  UNREFERENCED_PARAMETER(pvUser);

  int32_t    lRet   = 0;
  if (NXAPI_NO_ERROR == (lRet = nxDrvInit()))
  {
    CIFXHANDLE hNxApi = GetNxApiHandle(ptBoardInfo->tSystemInfo.ulSerialNumber, ptBoardInfo->tSystemInfo.ulDeviceNumber);

    if (NULL != hNxApi)
    {
      if (ulDownloadMode == DOWNLOAD_MODE_FIRMWARE)
        lRet = nxDrvDownload ( hNxApi, ulChannel, NXAPI_CMD_FIRMWARE, ulFileSize, pszFileName, pabFileData, NULL, NULL);
      else if (ulDownloadMode == DOWNLOAD_MODE_CONFIG)
        lRet = nxDrvDownload ( hNxApi, ulChannel, NXAPI_CMD_CONFIGURATION, ulFileSize, pszFileName, pabFileData, NULL, NULL);
      else
        lRet = CIFX_FILE_TYPE_INVALID;
    }
    else
    {
      lRet = CIFX_INVALID_HANDLE;
    }

    nxDrvExit();
  }

  return lRet;
}

/*****************************************************************************/
/*! Structure to track channel access                                        */
/*****************************************************************************/
typedef struct CHANNELTag
{
  unsigned long ulOpenCnt;
  CString       csName;
} CHANNEL_ACCESS_T;

std::map<CIFXHANDLE, CHANNEL_ACCESS_T> g_cmChannels;
static BOOL g_fMapChanged = true;

/*****************************************************************************/
/*! Structure holding name and width of the lists columns                    */
/*****************************************************************************/
static struct
{
  LPCTSTR       szColumn;
  unsigned char bWidth;
  unsigned long ulAlign;

} s_atColumns[] =
{
  { _T("Client IP") , 30, LVCFMT_LEFT },
  { _T("Host Name") , 40, LVCFMT_LEFT },
  { _T("RX [Bytes]"), 20, LVCFMT_LEFT },
  { _T("TX [Bytes]"), 20, LVCFMT_LEFT },
};

/*****************************************************************************/
/*! Writes the Fileverion in the version static field. hardcoded
/*****************************************************************************/
static void WriteVersion( CWnd* pcStaticCtrl)
{
  CFileVersionInfo cFileVersion;
  if(cFileVersion.Open(GetModuleHandle(NULL)))
  {
    CString csVersion;
    csVersion.Format(_T("Version: %d.%d.%d.%d" ),
                     cFileVersion.GetFileVersionMajor(),  // Major version
                     cFileVersion.GetFileVersionMinor(),  // Minor version
                     cFileVersion.GetFileVersionBuild(),  // Build number
                     cFileVersion.GetFileVersionQFE());   // QFE
    pcStaticCtrl->SetWindowText(csVersion);
  }
}

/*****************************************************************************/
/*! Writes the Fileverion in the version static field. hardcoded
/*****************************************************************************/
static void WriteMarshallerVersion( CWnd* pcStaticCtrl)
{
  CString csVersion;
  csVersion.Format(_T("(Marshaller Version: %d.%d.%d.%d)" ),
                    MARSHALLER_VERSION_MAJOR,  // Major version
                    MARSHALLER_VERSION_MINOR,  // Minor version
                    MARSHALLER_VERSION_BUILD,  // Build number
                    MARSHALLER_VERSION_REVISION);   // QFE
  pcStaticCtrl->SetWindowText(csVersion);
}

/*****************************************************************************/
/*! Function to set the Channel text.                                        */
/*****************************************************************************/
static void TrackChannelAccess_Open(CIFXHANDLE hChannel, char* szName)
{
  std::map<CIFXHANDLE, CHANNEL_ACCESS_T>::iterator iterChannelMap = g_cmChannels.find(hChannel);

  if (g_cmChannels.end() == iterChannelMap)
  {
    CHANNEL_ACCESS_T tChannel;
    tChannel.csName = szName;
    tChannel.ulOpenCnt = 1;
    g_cmChannels.insert(std::make_pair(hChannel, tChannel));
    g_fMapChanged = true;

  } else
  {
    iterChannelMap->second.ulOpenCnt++;
  }
}

/*****************************************************************************/
/*! Function to set the Channel text.                                        */
/*****************************************************************************/
static void TrackChannelAccess_Close(CIFXHANDLE hChannel)
{
  std::map<CIFXHANDLE, CHANNEL_ACCESS_T>::iterator iterChannelMap = g_cmChannels.find(hChannel);

  if (g_cmChannels.end() != iterChannelMap)
  {
    if (0 == --iterChannelMap->second.ulOpenCnt)
    {
      g_cmChannels.erase(iterChannelMap);
      g_fMapChanged = true;
    }
  }
}

/*****************************************************************************/
/*! Wrapper for xSysdeviceOpen to track sysdevice access
*   \param hDriver      Driver handle
*   \param szBoard      Name of the board to open
*   \param phSysdevice  Returned handle to the System device area
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
static int32_t APIENTRY xSysdeviceOpenWrap(CIFXHANDLE hDriver, char* szBoard, CIFXHANDLE* phSysdevice)
{
  int32_t lRet = CIFX_NO_ERROR;

  if (CIFX_NO_ERROR == (lRet = xSysdeviceOpen( hDriver, szBoard, phSysdevice)))
  {
    char               szName[MAX_PATH];
    sprintf(szName, "%s_SYS", szBoard);
    TrackChannelAccess_Open(*phSysdevice, szName);
  }

  return lRet;
}

/*****************************************************************************/
/*! Wrapper for xSysdeviceClose to track sysdevice access
*   \param hSysdevice  Handle to the System device to close
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
static int32_t APIENTRY xSysdeviceCloseWrap(CIFXHANDLE hSysdevice)
{
  int32_t lRet = CIFX_NO_ERROR;

  if (CIFX_NO_ERROR == (lRet = xSysdeviceClose( hSysdevice)))
  {
    TrackChannelAccess_Close(hSysdevice);
  }

  return lRet;
}

/*****************************************************************************/
/*! Wrapper for xChannelOpen to track channel access
*   \param hDriver    Driver handle
*   \param szBoard    DOS Device Name of the Board to open
*   \param ulChannel  Channel number to open (0..n)
*   \param phChannel  Returned handle to the channel (Needed for all channel
*                     specific operations)
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
static int32_t APIENTRY xChannelOpenWrap(CIFXHANDLE hDriver, char* szBoard, uint32_t ulChannel, CIFXHANDLE* phChannel)
{
  int32_t lRet = CIFX_NO_ERROR;

  if (CIFX_NO_ERROR == (lRet = xChannelOpen( hDriver, szBoard, ulChannel, phChannel)))
  {
    char               szName[MAX_PATH];
    sprintf(szName, "%s_ch%d", szBoard, ulChannel);
    TrackChannelAccess_Open(*phChannel, szName);
  }

  return lRet;
}

/*****************************************************************************/
/*! Wrapper for xChannelClose to track channel access
*   \param hChannel  Handle to the channel device to close
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
static int32_t APIENTRY xChannelCloseWrap(CIFXHANDLE hChannel)
{
  int32_t lRet = CIFX_NO_ERROR;

  if (CIFX_NO_ERROR == (lRet = xChannelClose(hChannel)))
  {
    TrackChannelAccess_Close(hChannel);
  }

  return lRet;
}

/*****************************************************************************/
/*! \class CAboutDlg
 * This is a dialog class used to display about information                  */
/*****************************************************************************/
class CAboutDlg : public CDialog
{
  public:
    CAboutDlg();
    enum { IDD = IDD_ABOUTBOX };

  protected:
    virtual void DoDataExchange(CDataExchange* pDX);    // DDX/DDV support

  protected:
    DECLARE_MESSAGE_MAP()

  public:
    virtual BOOL OnInitDialog();
};

/*****************************************************************************/
/*! This is the constructor of the CAboutDlg class                           */
/*****************************************************************************/
CAboutDlg::CAboutDlg() : CDialog(CAboutDlg::IDD)
{

}

/*****************************************************************************/
/*! This function is called to exchange and validate dialog data.
*   \param CDataExchange A pointer to a CDataExchange object.                */
/*****************************************************************************/
void CAboutDlg::DoDataExchange(CDataExchange* pDX)
{
  CDialog::DoDataExchange(pDX);
}

/*****************************************************************************/
/*! This function is called in response to the WM_INITDIALOG message.        */
/*****************************************************************************/
BOOL CAboutDlg::OnInitDialog()
{
  CDialog::OnInitDialog();

  WriteVersion(GetDlgItem(IDC_FILEVERSION));
  WriteMarshallerVersion(GetDlgItem(IDC_MARSHALLERVERSION));

  return TRUE;  // return TRUE unless you set the focus to a control
  // EXCEPTION: OCX Property Pages should return FALSE
}

BEGIN_MESSAGE_MAP(CAboutDlg, CDialog)
END_MESSAGE_MAP()

/*****************************************************************************/
/*! This is the constructor of the CcifXTCPServerDlg class                   */
/*****************************************************************************/
CcifXTCPServerDlg::CcifXTCPServerDlg(CWnd* pParent /*=NULL*/)
  : CDialog(CcifXTCPServerDlg::IDD, pParent)
{
  m_hIcon = AfxGetApp()->LoadIcon(IDR_MAINFRAME);
}

/*****************************************************************************/
/*! This function is called to exchange and validate dialog data.
*   \param CDataExchange A pointer to a CDataExchange object.                */
/*****************************************************************************/
void CcifXTCPServerDlg::DoDataExchange(CDataExchange* pDX)
{
  CDialog::DoDataExchange(pDX);
  DDX_Control(pDX, IDC_LST_CLIENTS, m_cClientListCtrl);
}

BEGIN_MESSAGE_MAP(CcifXTCPServerDlg, CDialog)
  ON_WM_SYSCOMMAND()
  ON_WM_PAINT()
  ON_MESSAGE(MARSHALLER_REQUEST, OnMarshallerRequest)
  ON_WM_QUERYDRAGICON()
  //}}AFX_MSG_MAP
  ON_WM_TIMER()
  ON_WM_DESTROY()
END_MESSAGE_MAP()

/*****************************************************************************/
/*! Initialization the Marshallar                                         */
/*****************************************************************************/
uint32_t CcifXTCPServerDlg::InitMarshaller()
{
  HIL_MARSHALLER_PARAMS_T           tParams        = {0};
  HIL_MARSHALLER_CONNECTOR_PARAMS_T tTCPConnector  = {0};
  uint32_t                          eRet           = HIL_MARSHALLER_E_SUCCESS;

  tTCPConnector.pfnConnectorInit = TCPConnectorInit;
  tTCPConnector.pvConfigData     = NULL;
  tTCPConnector.ulDataBufferCnt  = 5;
  tTCPConnector.ulDataBufferSize = 6000;
  tTCPConnector.ulTimeout        = 1000;

  TRANSPORT_LAYER_CONFIG_T          tCifXTransport = {0};
  CIFX_TRANSPORT_CONFIG             tCifXConfig    = {0};

  tCifXConfig.tDRVFunctions.pfnxDriverOpen                 = xDriverOpen;
  tCifXConfig.tDRVFunctions.pfnxDriverClose                = xDriverClose;
  tCifXConfig.tDRVFunctions.pfnxDriverGetInformation       = xDriverGetInformation;
  tCifXConfig.tDRVFunctions.pfnxDriverGetErrorDescription  = xDriverGetErrorDescription;
  tCifXConfig.tDRVFunctions.pfnxDriverEnumBoards           = xDriverEnumBoards;
  tCifXConfig.tDRVFunctions.pfnxDriverEnumChannels         = xDriverEnumChannels;
  tCifXConfig.tDRVFunctions.pfnxDriverMemoryPointer        = xDriverMemoryPointer;
  tCifXConfig.tDRVFunctions.pfnxDriverRestartDevice        = xDriverRestartDevice;
  tCifXConfig.tDRVFunctions.pfnxSysdeviceOpen              = xSysdeviceOpenWrap;
  tCifXConfig.tDRVFunctions.pfnxSysdeviceClose             = xSysdeviceCloseWrap;
  tCifXConfig.tDRVFunctions.pfnxSysdeviceReset             = xSysdeviceReset;
  tCifXConfig.tDRVFunctions.pfnxSysdeviceResetEx           = xSysdeviceResetEx;
  tCifXConfig.tDRVFunctions.pfnxSysdeviceGetMBXState       = xSysdeviceGetMBXState;
  tCifXConfig.tDRVFunctions.pfnxSysdevicePutPacket         = xSysdevicePutPacket;
  tCifXConfig.tDRVFunctions.pfnxSysdeviceGetPacket         = xSysdeviceGetPacket;
  tCifXConfig.tDRVFunctions.pfnxSysdeviceDownload          = xSysdeviceDownload;
  tCifXConfig.tDRVFunctions.pfnxSysdeviceInfo              = xSysdeviceInfo;
  tCifXConfig.tDRVFunctions.pfnxSysdeviceFindFirstFile     = xSysdeviceFindFirstFile;
  tCifXConfig.tDRVFunctions.pfnxSysdeviceFindNextFile      = xSysdeviceFindNextFile;
  tCifXConfig.tDRVFunctions.pfnxChannelOpen                = xChannelOpenWrap;
  tCifXConfig.tDRVFunctions.pfnxChannelClose               = xChannelCloseWrap;
  tCifXConfig.tDRVFunctions.pfnxChannelDownload            = xChannelDownload;
  tCifXConfig.tDRVFunctions.pfnxChannelGetMBXState         = xChannelGetMBXState;
  tCifXConfig.tDRVFunctions.pfnxChannelPutPacket           = xChannelPutPacket;
  tCifXConfig.tDRVFunctions.pfnxChannelGetPacket           = xChannelGetPacket;
  tCifXConfig.tDRVFunctions.pfnxChannelGetSendPacket       = xChannelGetSendPacket;
  tCifXConfig.tDRVFunctions.pfnxChannelConfigLock          = xChannelConfigLock;
  tCifXConfig.tDRVFunctions.pfnxChannelReset               = xChannelReset;
  tCifXConfig.tDRVFunctions.pfnxChannelInfo                = xChannelInfo;
  tCifXConfig.tDRVFunctions.pfnxChannelWatchdog            = xChannelWatchdog;
  tCifXConfig.tDRVFunctions.pfnxChannelHostState           = xChannelHostState;
  tCifXConfig.tDRVFunctions.pfnxChannelBusState            = xChannelBusState;
  tCifXConfig.tDRVFunctions.pfnxChannelIOInfo              = xChannelIOInfo;
  tCifXConfig.tDRVFunctions.pfnxChannelIORead              = xChannelIORead;
  tCifXConfig.tDRVFunctions.pfnxChannelIOWrite             = xChannelIOWrite;
  tCifXConfig.tDRVFunctions.pfnxChannelIOReadSendData      = xChannelIOReadSendData;
  tCifXConfig.tDRVFunctions.pfnxChannelControlBlock        = xChannelControlBlock;
  tCifXConfig.tDRVFunctions.pfnxChannelCommonStatusBlock   = xChannelCommonStatusBlock;
  tCifXConfig.tDRVFunctions.pfnxChannelExtendedStatusBlock = xChannelExtendedStatusBlock;
  tCifXConfig.tDRVFunctions.pfnxChannelPLCMemoryPtr        = xChannelPLCMemoryPtr;
  tCifXConfig.tDRVFunctions.pfnxChannelPLCIsReadReady      = xChannelPLCIsReadReady;
  tCifXConfig.tDRVFunctions.pfnxChannelPLCIsWriteReady     = xChannelPLCIsWriteReady;
  tCifXConfig.tDRVFunctions.pfnxChannelPLCActivateWrite    = xChannelPLCActivateWrite;
  tCifXConfig.tDRVFunctions.pfnxChannelPLCActivateRead     = xChannelPLCActivateRead;
  tCifXConfig.tDRVFunctions.pfnxChannelFindFirstFile       = xChannelFindFirstFile;
  tCifXConfig.tDRVFunctions.pfnxChannelFindNextFile        = xChannelFindNextFile;

  /* Install download hook */
  xDownloadHook_Install(&tCifXConfig.tDRVFunctions, HandleFileStorage, NULL);

  tCifXTransport.pfnInit  = cifXTransportInit;
  tCifXTransport.pvConfig = &tCifXConfig;

  tParams.ulMaxConnectors = 1;
  tParams.atTransports    = &tCifXTransport;
  tParams.ulTransportCnt  = 1;

  tParams.ptConnectors    = &tTCPConnector;
  tParams.ulConnectorCnt  = 1;


  if (HIL_MARSHALLER_E_SUCCESS == (eRet = HilMarshallerStart(&tParams, &m_pvMarshaller, MarshallerRequestWrapper, this)))
  {
    /* Create timer event to periodically call marshaller timer */
    SetTimer(IDT_CYCLIC, 10, NULL);

    /* Create timer event to refresh tcp information */
    SetTimer(IDT_TCPINFO, 500, NULL);
  }

  return eRet;
}

/*****************************************************************************/
/*! Destroy the Marshaller                                                   */
/*****************************************************************************/
void CcifXTCPServerDlg::DeinitMarshaller()
{
  /* Kill marshaller timer event */
  KillTimer(IDT_CYCLIC);

  /* Kill tcp information timer event */
  KillTimer(IDT_TCPINFO);


  if(NULL != m_pvMarshaller)
    HilMarshallerStop(m_pvMarshaller);

  /* Remove download hook */
  xDownloadHook_Remove();
}

/*****************************************************************************/
/*! This function is called in response to the WM_INITDIALOG message.        */
/*****************************************************************************/
BOOL CcifXTCPServerDlg::OnInitDialog()
{
  CRect     sRect;
  CPoint    sPoint(0,0);
  GetDlgItem(IDC_FRAME)->GetWindowRect(&sRect);
  ClientToScreen(&sPoint);

  sRect.left   -= sPoint.x;
  sRect.right  -= sPoint.x;
  sRect.top    -= sPoint.y;
  sRect.bottom -= sPoint.y;

  m_cClientListCtrl.CreateEx(WS_EX_CLIENTEDGE,TVS_HASLINES|TVS_LINESATROOT|TVS_FULLROWSELECT|WS_CHILD|WS_VISIBLE|WS_TABSTOP,sRect,this,1234);

  CDialog::OnInitDialog();

  WriteVersion(GetDlgItem(IDC_FILEVERSION));

  // Add "About..." menu item to system menu.

  // IDM_ABOUTBOX must be in the system command range.
  ASSERT((IDM_ABOUTBOX & 0xFFF0) == IDM_ABOUTBOX);
  ASSERT(IDM_ABOUTBOX < 0xF000);

  CMenu* pSysMenu = GetSystemMenu(FALSE);
  if (pSysMenu != NULL)
  {
    pSysMenu->AppendMenu(MF_SEPARATOR);
    pSysMenu->AppendMenu(MF_STRING, IDM_ABOUTBOX, _T("&About cifX TCP/IP Server..."));
    pSysMenu->AppendMenu(MF_STRING, IDM_HOSTINFO, _T("Show &Host Information"));
  }

  CRect rectList;
  m_cClientListCtrl.GetWindowRect(rectList);

  for(int iIdx = 0; iIdx < sizeof(s_atColumns) / sizeof(s_atColumns[0]); ++iIdx)
  {
    m_cClientListCtrl.InsertColumn(iIdx,
                                   s_atColumns[iIdx].szColumn,
                                   s_atColumns[iIdx].ulAlign,
                                   rectList.Width() / 100 * s_atColumns[iIdx].bWidth);
  }

  m_cClientListCtrl.SetExtendedStyle(m_cClientListCtrl.GetExtendedStyle() | LVS_EX_FULLROWSELECT | LVS_EX_GRIDLINES | TVS_EX_ITEMLINES|TVS_EX_SUBSELECT);

  uint32_t eRet = InitMarshaller();

  if(HIL_MARSHALLER_E_SUCCESS != eRet)
  {
    CString csTemp;
    char    szErrorText[MAX_PATH] = {0};
    CString csErrorText;

    if(CIFX_NO_ERROR == xDriverGetErrorDescription(eRet,
                                                   szErrorText,
                                                   sizeof(szErrorText)))
    {
      csErrorText = szErrorText;
    } else
    {
      csErrorText = _T("<no error description available>");
    }

    csTemp.Format(_T("Marshaller Startup Error!\n\n")
                  _T("%s (eRet = 0x%08X)"), csErrorText
                  , eRet);

    AfxMessageBox(csTemp);
    EndDialog(0);
  }

  // Set the icon for this dialog.  The framework does this automatically
  //  when the application's main window is not a dialog
  SetIcon(m_hIcon, TRUE);     // Set big icon

  return TRUE;  // return TRUE  unless you set the focus to a control
}

/*****************************************************************************/
/*! This function is called when the user selects a command from the Control
 *  menu, or when the user selects the Maximize or the Minimize button.
 *  \param nID    Specifies the type of system command requested
 *  \param lParam This parameter is not used for our command                 */
/*****************************************************************************/
void CcifXTCPServerDlg::OnSysCommand(UINT nID, LPARAM lParam)
{
  switch (nID & 0xFFF0)
  {
    case IDM_ABOUTBOX:
      {
        CAboutDlg dlgAbout;
        dlgAbout.DoModal();
      }
      break;

    case IDM_HOSTINFO:
      {
        HostInformation dlgHostInfo;
        dlgHostInfo.DoModal();
      }
      break;

    default:
      CDialog::OnSysCommand(nID, lParam);
      break;
  }
}

/*****************************************************************************/
/*! The function is called when Windows or an application makes a request
 *  to repaint a portion of an application's window. If you add a minimize
 *  button to your dialog, you will need the code below to draw the icon.
 *  For MFC applications using the document/view model, this is automatically
 *  done for you by the framework.                                           */
/*****************************************************************************/
void CcifXTCPServerDlg::OnPaint()
{
  if (IsIconic())
  {
    CPaintDC dc(this); // device context for painting

    SendMessage(WM_ICONERASEBKGND, reinterpret_cast<WPARAM>(dc.GetSafeHdc()), 0);

    // Center icon in client rectangle
    int cxIcon = GetSystemMetrics(SM_CXICON);
    int cyIcon = GetSystemMetrics(SM_CYICON);
    CRect rect;
    GetClientRect(&rect);
    int x = (rect.Width() - cxIcon + 1) / 2;
    int y = (rect.Height() - cyIcon + 1) / 2;

    // Draw the icon
    dc.DrawIcon(x, y, m_hIcon);
  }
  else
  {
    CDialog::OnPaint();
  }
}

/*****************************************************************************/
/*! This function is called by a minimized (iconic) window that does not
 *  have an icon defined for its class.
 *  \return Cursor or icon handle                                            */
/*****************************************************************************/
HCURSOR CcifXTCPServerDlg::OnQueryDragIcon()
{
  return static_cast<HCURSOR>(m_hIcon);
}

/*****************************************************************************/
/*! Marshaller Request Wrapper                                               */
/*****************************************************************************/
void CcifXTCPServerDlg::MarshallerRequestWrapper(void* /*pvMarshaller*/, void* pvUser)
{
  CcifXTCPServerDlg* pcDlg = reinterpret_cast<CcifXTCPServerDlg*>(pvUser);

  pcDlg->PostMessage(MARSHALLER_REQUEST);
}

/*****************************************************************************/
/*! Marshaller Timer                                                         */
/*****************************************************************************/
void CcifXTCPServerDlg::OnTimer(UINT_PTR nIDEvent)
{
  switch (nIDEvent)
  {
    case IDT_CYCLIC:
      HilMarshallerTimer(m_pvMarshaller);
      break;
    case IDT_TCPINFO:
      {
        TCP_CONN_INFO_T tTcpInfo = {0};

        if ( (HIL_MARSHALLER_E_SUCCESS != TCPConnectorInfo(&tTcpInfo)) ||
             (!tTcpInfo.fConnected) )
        {
          m_cClientListCtrl.DeleteItem(m_hClientItem);
          m_hClientItem = NULL;
        } else
        {
          TCHAR szBuffer[16] = {0};

          if ((g_fMapChanged) || (!m_hClientItem))
          {
            m_cClientListCtrl.DeleteItem(m_hClientItem);
            m_hClientItem = m_cClientListCtrl.InsertItem(CString(tTcpInfo.szIPAddress),TVI_ROOT);
            for( std::map<CIFXHANDLE, CHANNEL_ACCESS_T>::iterator iterChannelMap = g_cmChannels.begin();
                 iterChannelMap != g_cmChannels.end(); iterChannelMap++)
            {
              m_cClientListCtrl.InsertItem(iterChannelMap->second.csName, m_hClientItem);
            }
            m_cClientListCtrl.Expand(m_hClientItem, TVE_EXPAND);
            g_fMapChanged = false;
          }
          m_cClientListCtrl.SetItemText( m_hClientItem, CString(tTcpInfo.szHostname), SUBITEM_HOSTNAME);
          m_cClientListCtrl.SetItemText( m_hClientItem, _ltot(tTcpInfo.ulRxCount, szBuffer, 10), SUBITEM_RXCOUNT);
          m_cClientListCtrl.SetItemText( m_hClientItem, _ltot(tTcpInfo.ulTxCount, szBuffer, 10), SUBITEM_TXCOUNT);
        }
      }
      break;
    default:
      break;
  }
  CDialog::OnTimer(nIDEvent);
}

/*****************************************************************************/
/*! Marshaller Requests                                                      */
/*****************************************************************************/
LRESULT CcifXTCPServerDlg::OnMarshallerRequest(WPARAM /*wParam*/, LPARAM /*lParam*/)
{
  HilMarshallerMain(m_pvMarshaller);

  return TRUE;
}

/*****************************************************************************/
/*! This function is called to inform the TCP server dialog object that
*   it is being destroyed.                                                   */
/*****************************************************************************/
void CcifXTCPServerDlg::OnDestroy()
{
  CDialog::OnDestroy();

  DeinitMarshaller();
}
